unit uCreature;

interface

uses
    WinTypes,WinProcs,Graphics,Classes,IniFiles,
    uDuck, uParts, uLogoCanvas, SysUtils,
    uMutate, uSettings, Math, XMLDoc, XMLIntf;

const
   kCreatureStartCount = 1;
   kFoodStartCount = 300;
   ksGenPoolPath = '';
   ksAlivePath = 'Alive\';

type
    TCreature = class(TVirtualCreature)
      private
        FRadius : Single;
        FPhi : Single;
        FColor : TColor;
        FdX,FdY : Single;
        FXPos,FYPos, FdPhi : Single;
        FParts : TPartList;
        FMaxEnergy : Single;
        FbVegetable : Boolean;
        FsName : String;
        FbDebug : Boolean;
        FMutationChance : Single; {Kans op vergissing bij kopieren}
        procedure SetXPos( NewXPos : Single );
        procedure SetYPos( NewYPos : Single );
        function GetEnergy : Single;
        procedure SetEnergy(val : Single);
      public
        constructor Create;
        constructor CreateSimple;
        constructor CreateComplex;
        constructor CreateCloneAndDivide(Parent : TCreature);
        destructor  Destroy; override;
        procedure Step;
        procedure Draw( LogoCanvas : TLogoCanvas );
        function CostOfLiving : Single;
        function MutateName(S : String ) : String;
        function FindPartByID( ID : String ) : TPart; override;
        function DebugInfo : String;
        
        procedure SaveToFile(FN :String);
        procedure LoadFromFile(FN :String);

        property Radius : Single read FRadius write FRadius;
        property Energy : Single read GetEnergy write SetEnergy;
        property Color : TColor read FColor write FColor;
        property XPos : Single read FXPos write SetXPos;
        property YPos : Single read FYPos write SetYPos;
        property dX : Single read FdX write FdX;
        property dY : Single read FdY write FdY;
        property Phi : Single read FPhi write FPhi;
        property IsVegetable : Boolean read FbVegetable;
        property Name : String read FsName write FsName;
    end;

   TCreatureList = class(TList)
      private
        function GetItems(I:Integer) : TCreature;
      public
        function Add( ACreature : TCreature ) : TCreature;
        procedure FilterSee( X,Y : Single;
                             Phi, dPhi : Single;
                             iSelf : Integer;
                             var R,G,B : Single );
        property Items[ I : Integer ] : TCreature read GetItems;
        function FindCreatureAt(X,Y:Single) : TCreature;
        procedure UpdateSenses;
        procedure DoSteps;
        procedure CheckCollision( C1,C2 : TCreature );
        procedure Load( Width, Height : Integer; slFileNames : TStrings = nil );
        procedure Save( Path : String );
    end;


implementation

uses
    uGrid;

{ ___  _   _   _  _  ___      _   _ }
{  |  | ` |_) |_ /_\  |  | | |_) |_ }
{  |  |_, | \ |_ | |  |  |_| | \ |_ }

constructor TCreature.Create;
begin
    inherited Create;
    FdX := 0;
    FdY := 0;
    FParts := TPartList.Create(self);
    FMutationChance := 0.05;
end;

constructor TCreature.CreateSimple;
begin
    Create;
    Radius := 10;
    FMaxEnergy := 160;
    Color := $0200A000;
    FbVegetable := True;
    FMutationChance := 0.0;
    Name := 'APPEL';
end;

constructor TCreature.CreateComplex;
var
    Eye1, Eye2 : TEye;
    Eng1, Eng2, Eng3 : TEngine;
    Brain : TBrain;
    Mouth1,Mouth2 : TMouth;
begin
    Create;
    Name := 'ADAM';
    FbVegetable := False;
    Radius := 40;
    FMaxEnergy := 2000;
    Color := $02005050;
    Brain := FParts.Add( TBrain.Create(Self, 'Brain', 0.0, 0.0, 0.0, 1.0, '') ) as TBrain;
    Eye1 := FParts.Add( TEye.Create(Self, 'RightEye', Pi/6) ) as TEye;
    Eye2 := FParts.Add( TEye.Create(Self, 'LeftEye', -Pi/6) ) as TEye;
    Mouth1 := FParts.Add( TMouth.Create(Self, 'RightMouth', 0.18) ) as TMouth;
    Mouth2 := FParts.Add( TMouth.Create(Self, 'LeftMouth', -0.18) ) as TMouth;
    Eng1 := FParts.Add( TEngine.Create(Self, 'MidEngine', Pi,0, 'RightEye.1*8 + LeftEye.1*8') ) as TEngine;
    Eng2 := FParts.Add( TEngine.Create(Self, 'RightEngine', Pi-0.13,-0.13, 'LeftEye.1*8+Brain.0*5')) as TEngine;
    Eng3 := FParts.Add( TEngine.Create(Self, 'LeftEngine', Pi+0.13,0.13, 'RightEye.1*8')) as TEngine;
end;

constructor TCreature.CreateCloneAndDivide(Parent : TCreature);
var
    I : Integer;
begin
    Create;
    Parent.Energy := Parent.Energy/2;
    Energy := Parent.Energy;
    FbVegetable := Parent.FbVegetable;
    FMutationChance := Parent.FMutationChance;
    FPhi := Parent.FPhi+Pi;
    FColor := MutateColor( Parent.FColor, FMutationChance );
    FdX := -Parent.FdX;
    FdY := -Parent.FdY;
    FXPos := Parent.FXPos+FdX;
    FYPos := Parent.FYPos+FdY;
    FdPhi := Parent.FdPhi;
    Name := Parent.Name;
    FMaxEnergy := MutateReal(Parent.FMaxEnergy, FMutationChance);
    for I := 0 to Parent.FParts.Count-1 do
        FParts.Add( Parent.FParts.Items[I].CreateClone(Self) );
    for I := 0 to FParts.Count-1 do
        FParts[I].Mutate(FMutationChance);

    if (-Random<FMutationChance) and (FParts.Count<>0) then begin
        Name := MutateName(Name);
        if (Random(2)=0) and false then begin
            { Verwijder random lichaamsdeel:}
            I := Random(FParts.Count);
            FParts.Items[I].Free;
            FParts.Delete(I);
        end else begin
            { Voeg random lichaamsdeel toe:}
            I := Random(FParts.Count);
            FParts.Items[I].Phi := FParts.Items[I].Phi+0.1;
            FParts.Add( FParts.Items[I].CreateClone(Self, True) );
            FParts.Items[I].Phi := FParts.Items[I].Phi-0.2;
        end;
    end;
    PutCreature(Self, XPos,YPos);
end;

destructor TCreature.Destroy;
begin
    DeleteCreature(Self, XPos,YPos );
    FreeObjectsFromList( FParts );
    FParts.Free;
    inherited Destroy;
end;

procedure TCreature.SetXPos( NewXPos : Single );
begin
    if NewXPos=XPos then Exit;
    FXPos := NewXPos;
end;

procedure TCreature.SetYPos( NewYPos : Single );
begin
    if NewYPos=YPos then Exit;
    FYPos := NewYPos;
end;

function TCreature.GetEnergy : Single;
begin
    Result := Radius*Radius;
end;

procedure TCreature.SetEnergy(val : Single);
begin
    Radius := Sqrt(val);
end;

function TCreature.CostOfLiving : Single;
begin
    Result := Settings.CostOfLiving*(Settings.CreatureCost + Radius * Settings.RadiusCost + FParts.EnergyCost * Settings.PartsCost);
end;

procedure TCreature.Step;
var
    rRem : Single;
    oldX, oldY : Single;
begin
    oldX := XPos;
    oldY := YPos;
    XPos := XPos + FdX;
    YPos := YPos + FdY;
    Phi := Phi + FdPhi;
    rRem := 1-Settings.Remming* (Sqr( FdX) + Sqr(FdY));
    FdX := FdX*rRem + RandomZ(Round(Settings.Brown))/100;
    FdY := FdY*rRem + RandomZ(Round(Settings.Brown))/100;
    FdPhi := FdPhi*(1-Settings.PhiRemming) {+ RandomZ(kPhiBrown)/100};

    if FbVegetable then
        Energy := Energy + Settings.SunLight/(Sqr(FdX) + Sqr(FdY)+0.2) - CostOfLiving
    else
        Energy := Energy - CostOfLiving;
    MoveCreature( Self, oldX, oldY, XPos,YPos );
end;

procedure TCreature.Draw( LogoCanvas : TLogoCanvas );
var
    I : Integer;
begin
    with LogoCanvas do begin
        Brush.Color := Color;
        Pen.Color := $02B0B0B0;
        Circle( Radius );
        for I := 0 to FParts.Count-1 do begin
            PushStatus;
            Rotate( FParts.Items[I].Phi );
            Move( FParts.Items[I].R*Radius );
            Brush.Color := Color;
            Pen.Color := $02B0B0B0;
            Scale := Scale*Radius;
            FParts.Items[I].Draw( LogoCanvas );
            PopStatus;
        end;
    end;
end;

const kVowels : array[0..4] of Char = ( 'A', 'E', 'O', 'I', 'U' );
const kConsonant : array[0..17] of Char = ('B','C','D','F','G','H','J','K','L','M','N','P','R','S','T','V','W','Z');

function IsVowel(Ch : Char) : Boolean;
var
    I : Integer;
begin
    for I := Low(kVowels) to High(kVowels) do
        if kVowels[I] = Ch then begin
            Result := True;
            Exit;
        end;
    Result := False;
end;

function RandomVowel : Char;
begin
    Result := kVowels[Random(High(kVowels))+1];
end;

function RandomConsonant : Char;
begin
    Result := kConsonant[Random(High(kConsonant))+1];
end;

function TCreature.MutateName(S : String) : String;
var
    I : Integer;
begin
    if Random(Length(S))>7 then
        Delete(S, Random(Length(S)-2)+1,2)
    else if Random(14-Length(S))>7 then begin
        I := Random(Length(S))+1;
        if IsVowel(S[I]) then
            Insert(RandomVowel+RandomConsonant,S,I)
        else
            Insert(RandomConsonant+RandomVowel,S,I)
    end else begin
        I := Random(Length(S))+1;
        if IsVowel(S[I]) then
            S[I] := RandomVowel
        else
            S[I] := RandomConsonant;
    end;
    Result := S;
end;

procedure TCreature.LoadFromFile(FN: String);
var
    XML : IXMLDocument;
begin
    XML := LoadXMLDocument( FN );
    with XML.DocumentElement do begin
        FbVegetable := StrToBool(Attributes['vegetable']);
        Radius := StrToFloat(Attributes['radius']);
        FMaxEnergy := StrToFloat(Attributes['maxenergy']);
        Color := StrToInt(Attributes['color']);
    end;
    FParts.LoadFromXML(XML);
end;

procedure TCreature.SaveToFile(FN: String);
var
    XML : TXMLDocument;
begin
    XML := TXMLDocument.Create(nil);
    XML.Active := True;
    XML.AddChild('cambrium');
    with XML.DocumentElement do begin
        Attributes['vegetable'] := BoolToStr(FbVegetable,true);
        Attributes['radius']    := FloatToStr(Radius);
        Attributes['maxenergy'] := FloatToStr(FMaxEnergy);
        Attributes['color']     := IntToStr(Color);
    end;
    FParts.SaveToXML(XML);
    XML.SaveToFile(FN);
end;

function TCreature.FindPartByID(ID: String): TPart;
begin
    Result := FParts.FindPartByID(ID);
end;

function TCreature.DebugInfo: String;
begin
    if not FbDebug then Exit;
    if FindPartByID('MidEngine')=nil then begin
        Result := '';
        Exit;
    end;
    Result := Format( 'RightEye.Green = %f\nLeftEye.Green = %f\nMidEngine = %f, RightEngine = %f, LeftEngine = %f',
                      [FindPartByID('RightEye').ReadOutput(1),
                       FindPartByID('LeftEye').ReadOutput(1),
                       (FindPartByID('MidEngine') as TEngine).Speed,
                       (FindPartByID('RightEngine') as TEngine).Speed,
                       (FindPartByID('LeftEngine') as TEngine).Speed ]);
end;




{ ___  _   _   _  _  ___      _   _    _  _  ___ }
{  |  | ` |_) |_ /_\  |  | | |_) |_ |  | (_   |  }
{  |  |_, | \ |_ | |  |  |_| | \ |_ |_ |  _)  |  }

function TCreatureList.Add( ACreature : TCreature ) : TCreature;
begin
    inherited Add(ACreature);
    Result := ACreature;
end;

function TCreatureList.GetItems(I:Integer) : TCreature;
begin
    Result := TObject(inherited Items[I]) as TCreature;
end;

function TCreatureList.FindCreatureAt(X,Y:Single) : TCreature;
var
    I : Integer;
begin
    for I := 0 to Count-1 do begin
        if Sqr(X-Items[I].XPos)+Sqr(Y-Items[I].YPos) < Sqr(Items[I].Radius) then begin
            Result := Items[I];
            Exit;
        end;
    end;
    Result := nil;
end;

procedure TCreatureList.FilterSee( X,Y : Single;
                                   Phi, dPhi : Single;
                                   iSelf : Integer;
                                   var R,G,B : Single );
var
    Sn,Cs,
    Strl,
    dX,dY : Single;
    I : Integer;

begin
    Cs := Cos(Phi);
    Sn := Sin(Phi);
    dPhi := Sqr(dPhi);
    R := 0; G := 0; B := 0;
    for I := 0 to Count-1 do begin
        if I=iSelf then Continue;
        dX := Items[I].XPos-X;
        dY := Items[I].YPos-Y;
        Strl := Sqrt(Sqr(dX)+Sqr(dY));
        if (Strl=0) or (Sqr( (dX/Strl) - Cs) + Sqr( (dY/Strl) - Sn ) < dPhi) then begin
            Strl := 0.05/abs(Strl);
            R := R + Strl * GetRValue(Items[I].Color);
            G := G + Strl * GetGValue(Items[I].Color);
            B := B + Strl * GetBValue(Items[I].Color);
        end;
    end;
end;

procedure TCreatureList.UpdateSenses;
var
    rPush, rRotate : Single;
    P : Integer;
    Eye : TEye;
    Mouth : TMouth;
    Eng : TEngine;
    I : Integer;
    MaxThrough,
    R,G,B : Single;
    Straal,
    X,Y : Single;
    Cre : TCreature;
begin
    for I := 0 to Count-1 do with Items[I] do begin
        for P := 0 to FParts.Count-1 do begin
            if (FParts.Items[P] is TEye) then begin
                Eye := FParts.Items[P] as TEye;
                FilterSee( XPos,YPos,
                           Eye.Phi+Phi, Pi/6,
                           I,
                           R,G,B );
                Eye.SetColors(R,G,B);
            end else if (FParts.Items[P] is TEngine) then begin
                Eng := FParts.Items[P] as TEngine;
                Eng.GetEffect( rPush, rRotate,
                               Radius  );
                FdX := FdX - Cos(Eng.Phi+Phi) * rPush;
                FdY := FdY - Sin(Eng.Phi+Phi) * rPush;
                FdPhi := FdPhi + rRotate;
            end else if (FParts.Items[P] is TBrain) then begin
                // do nothing
            end else if (FParts.Items[P] is TMouth) then begin
                Mouth := FParts.Items[P] as TMouth;
                Straal := Radius*(1+0.7*Mouth.Size);
                X := XPos + Cos(Mouth.Phi+Phi) * Straal;
                Y := YPos + Sin(Mouth.Phi+Phi) * Straal;
                Cre := FindCreatureAt(X,Y);
                MaxThrough := Energy/1000; {Maximaal 0.5% groei}
                if Cre<>nil then begin
                    if Cre.Energy<MaxThrough then
                        MaxThrough := Cre.Energy;
                    Energy := Energy + MaxThrough;
                    Cre.Energy := Cre.Energy - MaxThrough;
                    Mouth.Feeding(500*MaxThrough/Energy);
                end else
                    Mouth.Feeding(0);
            end;
        end;
    end;
    for I := 0 to Count-1 do with Items[I] do begin
        for P := 0 to FParts.Count-1 do begin
            FParts[P].Recalculate;
        end;
    end;
end;

procedure TCreatureList.CheckCollision( C1,C2 : TCreature );

    procedure MoveFromCenter( C : TCreature; CenX,CenY,F : Single );
    var
        _dX,_dY : Single;
        oldX, oldY : Single;
    begin
        _dX := F*(C.XPos - CenX);
        _dY := F*(C.YPos - CenY);
        oldX := C.XPos;
        oldY := C.YPos;
        C.XPos := oldX + _dX;
        C.YPos := oldY + _dY;
        MoveCreature( C, oldX, oldY, C.XPos, C.YPos );
    end;
var
    avdX,avdY : Single;
    CenX,CenY : Single;
    F, W1,W2 : Single;
    NewD, OldD : Single;

begin
    if (C1=nil) or (C2=nil) then
        C1:=C2;
    if Sqr(C1.XPos-C2.XPos) + Sqr(C1.YPos-C2.YPos) <
       Sqr(C1.Radius+C2.Radius) then begin
          W1 := C1.Energy/(C1.Energy+C2.Energy);
          W2 := 1-W1;
          avdX := W1*C1.dX + W2*C2.dX;
          avdY := W1*C1.dY + W2*C2.dY;
          W1 := C1.Radius/(C1.Radius+C2.Radius);
          W2 := 1-W1;
          C1.dX := avdX; C1.dY := avdY;
          C2.dX := avdX; C2.dY := avdY;
          CenX := C1.XPos*W2+ C2.XPos*W1;
          CenY := C1.YPos*W2+ C2.YPos*W1;
          NewD := C1.Radius + C2.Radius;
          OldD := Sqrt(Sqr(C1.XPos-C2.XPos) + Sqr(C1.YPos-C2.YPos));
          if OldD<0.001 then OldD := 0.001;
          F := (NewD/OldD)-0.99;
          MoveFromCenter( C1, CenX,CenY, F );
          MoveFromCenter( C2, CenX,CenY, F );
    end;
end;

procedure TCreatureList.DoSteps;

var
    I : Integer;
    AChild : TCreature;
    MaxR : Single;
begin
    UpdateSenses;

    MaxR := 0;
    for I := 0 to Count-1 do begin
        Items[I].Step;
        if Items[I].Radius>MaxR then
            MaxR := Items[I].Radius;
    end;

    {The cycle of life:}
    for I := Count-1 downto 0 do begin
        with Items[I] do begin
            if Energy<Settings.MinEnergy then begin
              try
                Items[I].Free;
                Delete(I);
              except
                MaxR := I;
              end;
            end else if Energy>FMaxEnergy then begin
                 if (Count<Settings.MaxFood) or (not IsVegetable) then begin
                     AChild := TCreature.CreateCloneAndDivide(Items[I]);
                     Add( AChild );
                 end else begin
                     Energy := FMaxEnergy;
                 end;
            end;
        end;
    end;

    { Check for collisions:}
    for I := Count-1 downto 0 do
        with Items[I] do
            FindCreatures(Items[I], XPos-MaxR-Radius,
                                    YPos-MaxR-Radius,
                                    XPos+MaxR+Radius,
                                    YPos+MaxR+Radius,
                                    CheckCollision ) ;

    {Apply gravity:}
    for I := Count-1 downto 0 do with Items[I] do begin
        dX := dX - XPos*Settings.Gravity;
        dY := dY - YPos*Settings.Gravity;
    end;
end;

const
    ExtDigs = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';

procedure TCreatureList.Save( Path : String );
var sName : String;
    I,J : Integer;
    slFiles : TStringList;
    sAlive : String;

begin
    if (Length(Path)>0) and (Path[Length(Path)]<>'\') then
        Path := Path + '\';
    for I := 0 to Count-1 do if not Items[I].IsVegetable then begin
        if FileExists( Path + Items[I].Name + '.cam' ) then Continue;
        Items[I].SaveToFile( Path + Items[I].Name + '.cam' );
    end;
end;

procedure TCreatureList.Load( Width, Height : Integer; slFileNames : TStrings );
var
    ACreature : TCreature;
    i,j : Integer;
    NumCreatures, NumFood : Integer;
    NumFiles, NumToCreate : Integer;
    CreatureFileName : String;
    PopIni : TIniFile;
    slFiles : TStringList;
    RandomFile : Integer;
    sGenPool, sAlive : String;
begin
    sGenPool := ksGenPoolPath + CreatureFileName;
    sAlive := ksAlivePath;

    PopIni := TIniFile.Create( 'population.ini' );
    NumCreatures := PopIni.ReadInteger('Population', 'NumCreatures', kCreatureStartCount);
    NumFood := PopIni.ReadInteger('Population', 'NumFood', kFoodStartCount);
    PopIni.Free;

    for i := 0 to NumCreatures-1 do begin
        if (slFileNames=nil) or (slFileNames.Count=0) then
            ACreature := TCreature.CreateComplex
        else begin
            ACreature := TCreature.Create;
            ACreature.LoadFromFile('animals\'+slFileNames[i mod slFileNames.Count])
        end;
        Add( ACreature );
        ACreature.XPos := Random(Width div 2) - Width div 4;
        ACreature.YPos := Random(Height div 2) - Height div 4;
    end;

    for j := 1 to NumFood do begin
        ACreature := TCreature.CreateSimple;
        Add( ACreature );
        ACreature.XPos := Random(Width div 2) - Width div 4;
        ACreature.YPos := Random(Height div 2) - Height div 4;
    end;
end;

begin
    Randomize;
end.


