unit uLogoCanvas;

interface

uses
    Classes,
    Graphics;

type
    TLogoStatus = class
        FPenColor : TColor;
        FBrushColor : TColor;
        FCurX, FCurY, FScale : Single;
        FCos : Single;
        FSin : Single;
    end;

    TLogoStatusList = class(TList)
      private
        function GetItems(I:Integer) : TLogoStatus;
      public
        function Add( ALogoStatus : TLogoStatus ) : TLogoStatus;
        property Items[ I : Integer ] : TLogoStatus read GetItems; 
        procedure Push( ALogoStatus : TLogoStatus );
        function Pop : TLogoStatus;
    end;

     TLogoCanvas = class
       private
         FCanvas : TCanvas;
         FCos : Single;
         FSin : Single;
         FCurX, FCurY : Single;
         FScale : Single;
         FStatusStack : TLogoStatusList;
         function GetBrush : TBrush;
         function GetPen : TPen;
         function GetFont : TFont;
       protected
         procedure MoveCurXY( ADist : Single );
       public
         constructor Create( ACanvas : TCanvas );
         destructor Destroy; override;
         procedure PushStatus;
         procedure PopStatus;
         procedure Rotate( APhi : Single );
         procedure Draw( ADist : Single );
         procedure Move( ADist : Single );
         procedure MoveAbs( AX,AY : Single );
         procedure RotateAbs( APhi : Single );
         procedure Circle( R : Single );
         procedure SetPos( AX,AY, APhi, AScale : Single );
         property Scale : Single read FScale write FScale;
         property Brush : TBrush read GetBrush;
         property Pen : TPen read GetPen;
         property Font : TFont read GetFont;
     end;

implementation


function TLogoStatusList.Add( ALogoStatus : TLogoStatus ) : TLogoStatus;
begin
    inherited Add(ALogoStatus);
    Result := ALogoStatus;
end;

function TLogoStatusList.GetItems(I:Integer) : TLogoStatus;
begin
    Result := TObject(inherited Items[I]) as TLogoStatus;
end;

procedure TLogoStatusList.Push( ALogoStatus : TLogoStatus );
begin
    Add( ALogoStatus );
end;

function TLogoStatusList.Pop : TLogoStatus;
begin
    Result := Items[Count-1];
    Delete(Count-1);
end;

{Logo Canvas zelf:}

constructor TLogoCanvas.Create( ACanvas : TCanvas );
begin
    FCanvas := ACanvas;
    FScale := 1.0;
    FStatusStack := TLogoStatusList.Create;
end;

destructor TLogoCanvas.Destroy;
begin
    FStatusStack.Free;
end;

procedure TLogoCanvas.PushStatus;
var
    AStatus : TLogoStatus;
begin
    AStatus := TLogoStatus.Create;
    AStatus.FPenColor := Pen.Color;
    AStatus.FBrushColor := Brush.Color;

    AStatus.FCurX := FCurX;
    AStatus.FCurY := FCurY;
    AStatus.FScale := FScale;
    AStatus.FCos := FCos;
    AStatus.FSin := FSin;
    FStatusStack.Push( AStatus );
end;

procedure TLogoCanvas.PopStatus;
var
    AStatus : TLogoStatus;
begin
    AStatus := FStatusStack.Pop;
    Pen.Color := AStatus.FPenColor;
    Brush.Color := AStatus.FBrushColor;
    FCurX := AStatus.FCurX;
    FCurY := AStatus.FCurY;
    FScale := AStatus.FScale;
    FCos := AStatus.FCos;
    FSin := AStatus.FSin;
    AStatus.Free;
end;

function TLogoCanvas.GetBrush : TBrush;
begin
    Result := FCanvas.Brush;
end;

function TLogoCanvas.GetPen : TPen;
begin
    Result := FCanvas.Pen;
end;

function TLogoCanvas.GetFont : TFont;
begin
    Result := FCanvas.Font;
end;

procedure TLogoCanvas.Rotate( APhi : Single );
var
    NewS, NewC, ACos, ASin : Single;
begin
    ACos := Cos(APhi);
    ASin := Sin(APhi);
    NewC := FCos * ACos - FSin * ASin;
    NewS := FSin * ACos + FCos * ASin;
    FCos := NewC;
    FSin := NewS;
end;

procedure TLogoCanvas.RotateAbs( APhi : Single );
begin
    FCos := Cos(APhi);
    FSin := Sin(APhi);
end;

procedure TLogoCanvas.MoveCurXY( ADist : Single );
begin
    FCurX := FCurX + FCos * ADist * FScale;
    FCurY := FCurY + FSin * ADist * FScale;
end;

procedure TLogoCanvas.Draw( ADist : Single );
begin
    MoveCurXY( ADist );
    FCanvas.LineTo( Round(FCurX), Round(FCurY) );
end;

procedure TLogoCanvas.Move( ADist : Single );
begin
    MoveCurXY( ADist );
    FCanvas.MoveTo( Round(FCurX), Round(FCurY) );
end;

procedure TLogoCanvas.MoveAbs( AX,AY : Single );
begin
    FCurX := AX;
    FCurY := AY;
    FCanvas.MoveTo( Round(FCurX), Round(FCurY) );
end;

procedure TLogoCanvas.Circle( R : Single );
begin
    { Een geroteeerde circel blijft een circel:}
    FCanvas.Ellipse( Round(FCurX + R*Scale),
                     Round(FCurY + R*Scale),
                     Round(FCurX - R*Scale),
                     Round(FCurY - R*Scale) );
end;

procedure TLogoCanvas.SetPos( AX,AY, APhi, AScale : Single );
begin
    MoveAbs(AX,AY);
    RotateAbs( APhi );
    Scale := AScale;
end;


end.
