unit uWorld;

interface

uses
    uProgram, Math, uAnimal, {uTool,} Classes, uConst, Windows;

type
    TSmell = record
        SmellPrint : Integer;
        Strength : Integer;
    end;

    TSmells = array[0..kMaxSmells] of TSmell;
    TGridCell = record
        FFood   : Integer;
        FAnimal : TAnimal;
        FSmells : TSmells;
    end;

    TMicroCodeMethod = procedure( An : TAnimal; P1,P2,P3 : Integer ) of object;

    TWorld = class
  private
    function GetGenCount: Integer;
      public
        FGrid : array[0..kMaxX-1,0..kMaxY] of TGridCell;
        FMicroCode : array[0..kMaxInstruction-1] of TMicroCodeMethod;
        FAnimalList : TAnimalList;
        FGenCount : Double;
      protected
        procedure MoveForward( An : TAnimal; P1,P2,P3 : Integer );
        procedure MoveBackward( An : TAnimal; P1,P2,P3 : Integer );
        procedure TurnLeft( An : TAnimal; P1,P2,P3 : Integer );
        procedure TurnRight( An : TAnimal; P1,P2,P3 : Integer );
        procedure TurnRandom( An : TAnimal; P1,P2,P3 : Integer );
        procedure Smell( An : TAnimal; SmellsLike, RegIdx, Distance : Integer );
        procedure Fork( An : TAnimal; Percentage, P2, P3 : Integer );
        procedure Eat( An : TAnimal; P1,P2,P3 : Integer );
        procedure TestFood(     An : TAnimal; dLeft, dForward, RegIdx  : Integer );
        procedure TestAnimal(   An : TAnimal; dLeft, dForward, RegIdx  : Integer );
        procedure TestObstacle( An : TAnimal; dLeft, dForward, RegIdx  : Integer );
        procedure BrancheNonEqual( An : TAnimal; RegIdx, CompareValue, RelJump : Integer );
        procedure BrancheGreater( An : TAnimal; RegIdx, CompareValue, RelJump : Integer );
        procedure AddRegs( An : TAnimal; DestReg, Source1, Source2 : Integer );
        procedure SubsRegs( An : TAnimal; DestReg, Source1, Source2 : Integer );
        procedure MultRegs( An : TAnimal; DestReg, Source1, Source2 : Integer );
        procedure DivRegs( An : TAnimal; DestReg, Source1, Source2 : Integer );
        procedure LoadConst( An : TAnimal; DestReg, Value, P3 : Integer );
        procedure LoadReg( An : TAnimal; DestReg, SrcReg, P3 : Integer );
        procedure Jump( An : TAnimal; RelJump, P2, P3 : Integer );
        procedure Loop( An : TAnimal; RegIdx, RelJump, P3 : Integer );
        procedure LoadEnergy( An : TAnimal; RegIdx, P2, P3 : Integer );
        procedure Hit( An : TAnimal; Power, P2,P3 : Integer );
      protected
        procedure MoveAnimal( An : TAnimal; dX,dY : Integer);
      public
        constructor Create;
        destructor Destroy; override;
        procedure Cycle;
        procedure FoodBlob;
        function CreateAnimal( Script : TStrings ) : TAnimal;
        property GenCount : Integer read GetGenCount;
    end;

implementation

{ TWorld }

procedure TWorld.AddRegs(An: TAnimal; DestReg, Source1, Source2: Integer);
begin
    with An do
        Registers[DestReg] := Registers[Source1] + Registers[Source2];
end;

procedure TWorld.BrancheGreater(An: TAnimal; RegIdx, CompareValue, RelJump : Integer);
begin
    with An do
        if Registers[RegIdx]>CompareValue then
            Jump( An, RelJump, 0,0 );
end;

procedure TWorld.BrancheNonEqual(An: TAnimal; RegIdx, CompareValue, RelJump : Integer);
begin
    with An do
        if Registers[RegIdx]<>CompareValue then
            Jump( An, RelJump, 0,0 );
end;

constructor TWorld.Create;
var
    I : Integer;
    X,Y : Integer;
begin
    for X:=0 to kMaxX-1 do
        FoodBlob;
        
    for I := 0 to kMaxObjects do
        FGrid[Random(kMaxX), Random(kMaxY)].FFood :=-1;

    for I := 0 to kExtendedPixels do begin
        repeat
             X :=Random(kMaxX);
             Y := Random(kMaxY);
        until FGrid[X,Y].FFood =-1;
        if random(2)=1 then
            X := (X+kMaxX+Random(2)*2-1) mod kMaxX
        else
            Y := (Y+kMaxY+Random(2)*2-1) mod kMaxY;
        FGrid[X,Y].FFood := -1;
    end;

    FAnimalList := TAnimalList.Create;
    FMicroCode[00]:= MoveForward;
    FMicroCode[01]:= MoveBackward;
    FMicroCode[02]:= TurnLeft;
    FMicroCode[03]:= TurnRight;
    FMicroCode[04]:= TurnRandom;
    FMicroCode[05]:= Smell;
    FMicroCode[06]:= Fork;
    FMicroCode[07]:= Eat;
    FMicroCode[08]:= TestFood;
    FMicroCode[09]:= TestAnimal;
    FMicroCode[10]:= TestObstacle;
    FMicroCode[11]:= BrancheNonEqual;
    FMicroCode[12]:= BrancheGreater;
    FMicroCode[13]:= AddRegs;
    FMicroCode[14]:= SubsRegs;
    FMicroCode[15]:= MultRegs;
    FMicroCode[16]:= DivRegs;
    FMicroCode[17]:= LoadConst;
    FMicroCode[18]:= LoadReg;
    FMicroCode[19]:= Jump;
    FMicroCode[20]:= Loop;
    FMicroCode[21]:= LoadEnergy;
    FMicroCode[22]:= Hit;
end;

function TWorld.CreateAnimal(Script: TStrings)  : TAnimal;
var
    X,Y : Integer;
begin
    repeat
        X := Random(kMaxX);
        Y := Random(kMaxY);
    until FGrid[X,Y].FAnimal=nil;
    FGrid[X,Y].FAnimal := TAnimal.Create( Script, 200, X,Y );
    FAnimalList.Add(FGrid[X,Y].FAnimal);
    Result := FGrid[X,Y].FAnimal;
end;


destructor TWorld.Destroy;
var
    I : Integer;
begin
    for I:=0 to FAnimalList.Count-1 do
        FAnimalList[I].Free;
    FAnimalList.Free;
    inherited;
end;

procedure TWorld.DivRegs(An: TAnimal; DestReg, Source1, Source2: Integer);
var
    A,B : Integer;
begin
    with An do begin
        A := Registers[Source1];
        B := Registers[Source2];
        if B<>0 then
            Registers[DestReg] := A div B;
    end;
end;

procedure TWorld.Eat(An: TAnimal; P1, P2, P3: Integer);
var
    ToEat : Integer;
begin
    with An do begin
        ToEat := Min( FGrid[FXPos,FYPos].FFood, kMaxEat );
        Inc(FEnergy,ToEat);
        Dec( FGrid[FXPos,FYPos].FFood, ToEat );
        FWaitCount:= kWait_Eat;
    end;
end;

procedure TWorld.Fork(An: TAnimal; Percentage, P2, P3: Integer);
var
    Child : TAnimal;
begin
    if Percentage <kMinForkPercentage then
        Percentage := kMinForkPercentage;
    if Percentage >kMaxForkPercentage then
        Percentage := kMaxForkPercentage;
    FGenCount := FGenCount + 1/FAnimalList.Count;
    Child := TAnimal.Clone(An);
    An.FEnergy := An.FEnergy * kSplitEfficiency div 100;
    Child.FEnergy := An.FEnergy * Percentage div 100;
    An.FEnergy := An.FEnergy * (100-Percentage) div 100;

    Child.FdX := -Child.Fdx;
    Child.FdY := -Child.FdY;
    MoveForward(Child,0,0,0);
    FAnimalList.Add( Child );

    An.FWaitCount := kWait_Fork;
    Child.FWaitCount := kWait_Fork;
end;

procedure TWorld.Hit(An: TAnimal; Power, P2, P3: Integer);
var
    X,Y : Integer;
    iRes : Integer;
begin
    X := (An.FXPos + An.FdX + kMaxX) mod kMaxX;
    Y := (An.FYPos + An.FdY + kMaxY) mod kMaxY;
    if FGrid[X,Y].FAnimal=nil then
        iRes:=-1
    else begin
        if Random(100)<101 then begin
            Inc(FGrid[X,Y].FFood, FGrid[X,Y].FAnimal.FEnergy );
            FGrid[X,Y].FAnimal.FEnergy := -1000;
        end;
        An.FWaitCount := kWait_HitSucces;
    end;
end;

procedure TWorld.Jump(An: TAnimal; RelJump, P2, P3: Integer);
begin
    with An do
        PC := PC + RelJump-1;
end;

procedure TWorld.LoadConst(An: TAnimal; DestReg, Value, P3: Integer);
begin
    with An do
        Registers[DestReg] := Value;
end;

procedure TWorld.LoadEnergy(An: TAnimal; RegIdx, P2, P3: Integer);
begin
    with An do
        Registers[RegIdx] := FEnergy;
end;

procedure TWorld.LoadReg(An: TAnimal; DestReg, SrcReg, P3: Integer);
begin
    with An do
        Registers[DestReg] := Registers[SrcReg];
end;

procedure TWorld.Loop(An: TAnimal; RegIdx, RelJump, P3: Integer);
begin
    with An do begin
        Registers[RegIdx] := Registers[RegIdx] -1;
        if Registers[RegIdx]<=0 then
            PC := PC + RelJump;
    end;
end;

procedure TWorld.MoveAnimal(An: TAnimal; dX, dY: Integer);
var
    X,Y : Integer;
begin
    with An do begin
        X := FXPos + dX;
        Y := FYPos + dY;
        if X<0 then X := kMaxX-1;
        if X>=kMaxX then X := 0;
        if Y<0 then Y := kMaxY-1;
        if Y>=kMaxY then Y := 0;
        if FGrid[X,Y].FAnimal<>nil then
            Exit;
        if FGrid[X,Y].FFood<0 then begin
            Exit;
        end;
        FGrid[X,Y].FAnimal := An;
        FGrid[FXPos,FYPos].FAnimal := nil;
        FXPos := X;
        FYPos := Y;
        FWaitCount := kWait_Move;
    end;
end;

procedure TWorld.MoveBackward(An: TAnimal; P1, P2, P3: Integer);
begin
    with An do begin
        MoveAnimal(An, -FdX, -FdY);
    end;
end;

procedure TWorld.MoveForward(An: TAnimal; P1, P2, P3: Integer);
begin
    with An do begin
        MoveAnimal(An, FdX, FdY);
    end;
end;

procedure TWorld.MultRegs(An: TAnimal; DestReg, Source1, Source2: Integer);
begin
    with An do
        Registers[DestReg] := Registers[Source1] * Registers[Source2];
end;

procedure TWorld.Smell(An: TAnimal; SmellsLike, RegIdx, Distance: Integer);
begin

end;

procedure TWorld.SubsRegs(An: TAnimal; DestReg, Source1, Source2: Integer);
begin
    with An do
        Registers[DestReg] := Registers[Source1] - Registers[Source2];
end;

procedure TWorld.TestAnimal(An: TAnimal; dLeft, dForward,RegIdx: Integer);
var
    X,Y : Integer;
    Col1,Col2, iRes : Integer;
begin
    with An do begin
        ToXY(dLeft, dForward, X,Y);
        Registers[RegIdx] := FGrid[X,Y].FFood;
        if FGrid[X,Y].FAnimal=nil then
            iRes:=-1
        else begin
            Col1 := FGrid[X,Y].FAnimal.Color shr 24;
            Col2 := An.Color shr 24;
            iRes := abs(Col1-Col2);
        end;
        Registers[RegIdx] := iRes;
    end;
end;

procedure TWorld.TestFood(An: TAnimal; dLeft, dForward,RegIdx: Integer);
var
    X,Y : Integer;
begin
    with An do begin
        ToXY(dLeft, dForward, X,Y);
        if FGrid[X,Y].FFood<=0 then
            Registers[RegIdx] := 0
        else
            Registers[RegIdx] := FGrid[X,Y].FFood;
    end;
end;

procedure TWorld.TestObstacle(An: TAnimal; dLeft, dForward,RegIdx: Integer);
var
    X,Y : Integer;
begin
    with An do begin
        ToXY(dLeft, dForward, X,Y);
        if FGrid[X,Y].FFood<=0 then
            Registers[RegIdx] := 1
        else
            Registers[RegIdx] := 0;
    end;
end;

procedure TWorld.TurnLeft(An: TAnimal; P1, P2, P3: Integer);
var
    Tmp : Integer;
begin
    with An do begin
        Tmp := FdX;
        FdX := FdY;
        FdY := -Tmp;
        FWaitCount := kWait_Turn;
    end;
end;

procedure TWorld.TurnRandom(An: TAnimal; P1, P2, P3: Integer);
begin
    if Random(2)=0 then
        TurnLeft(An,P1,P2,P3)
    else
        TurnRight(An,P1,P2,P3);
end;

procedure TWorld.TurnRight(An: TAnimal; P1, P2, P3: Integer);
var
    Tmp : Integer;
begin
    with An do begin
        Tmp := FdX;
        FdX := -FdY;
        FdY := Tmp;
        FWaitCount := kWait_Turn;
    end;
end;

procedure TWorld.Cycle;
var
    I : Integer;
    Method : TMicroCodeMethod;
begin
    // Distribute new food in one blob
    if Random(kDistributeFoodChange)=0 then 
        FoodBlob;
    // let all animals make their move:

    for I := 0 to FAnimalList.Count-1 do begin
        with FAnimalList[I] do begin
            if FWaitCount>0 then
                Dec(FWaitCount)
            else begin
                if FDebug>0 then
                    FDebug := FDebug;
                with FAnimalList[I].FProgram.FInstructions[FPC] do begin
                    Method := FMicroCode[Operation];
                    FWaitCount := 0;
                    Method(FAnimalList[I],Param1,Param2,Param3);
                end;
                Inc(FPC);
                if FPC>=FProgram.FSize then
                    FPC:=0
            end;
        end;
    end;

    // Kill animals with energy <0
    for I := FAnimalList.Count-1 downto 0 do begin
        if I<30 then
            FAnimalList[I].FEnergy:=FAnimalList[I].FEnergy*(2-1);
        Dec(FAnimalList[I].FEnergy,kCostOfLiving);
        if (FAnimalList[I].FEnergy<kInstructionCost*FAnimalList[I].FProgram.FSize) or
           (Random(kDeathFromIllnessChance)=0) then begin
            Inc(FGrid[FAnimalList[I].FXPos,FAnimalList[I].FYPos].FFood, FAnimalList[I].FEnergy);
            FGrid[FAnimalList[I].FXPos,FAnimalList[I].FYPos].FAnimal:=nil;
            FAnimalList[I].Free;
            FAnimalList.Delete(I);
        end;
    end;

end;

procedure TWorld.FoodBlob;
var
    iPixels, SQ, X,Y : Integer;
    iAmount, X1,Y1,X2,Y2 : Integer;
begin
    iPixels := kDistributeFoodChange * kFoodGrowth div (kMaxPlant div 2);
    X := Random(kMaxX);
    Y := Random(kMaxY);
    iPixels := Round(Sqrt(iPixels)) div 2;
    for X1 := X-iPixels to X+iPixels do begin
        X2 := (X1+kMaxX) mod kMaxX;
        SQ := Round(SQRT( iPixels*iPixels - Sqr(X1-X) ));
        for Y1 := Y-SQ to Y+SQ do begin
            Y2 := (Y1+kMaxY) mod kMaxY;
            iAmount := Round(kMaxPlant * (1-(abs(X1-X) + abs(Y1-Y))/iPixels));
            if iAmount<1 then iAmount :=1;
            if FGrid[X2,Y2].FFood>=0 then begin
                Inc( FGrid[X2,Y2].FFood, iAmount);
                if FGrid[X2,Y2].FFood>kMaxPlant then
                   FGrid[X2,Y2].FFood:=kMaxPlant
            end;
        end;
    end;
end;

function TWorld.GetGenCount: Integer;
begin
    GetGenCount := Round(FGenCount);
end;

end.
